// Package websocket is a chat client based on websocket.
package websocket

import (
	v20231130 "chat-demo-client-golang/v20231130"
	"context"
	"encoding/json"
	"fmt"
	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common"
	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common/profile"
	"log"
	"sync"
	"sync/atomic"
	"time"

	"github.com/davecgh/go-spew/spew"
	"github.com/gorilla/websocket"
	"github.com/zishang520/engine.io-go-parser/packet"
	eioparser "github.com/zishang520/engine.io-go-parser/parser"
	"github.com/zishang520/engine.io-go-parser/types"
	sioparser "github.com/zishang520/socket.io-go-parser/v2/parser"

	"chat-demo-client-golang/event"
)

// Client 会话客户端
type Client interface {
	// GetToken 获取连接令牌
	GetToken(ctx context.Context, req *GetTokenReq) (string, error)
	// Send 发送消息
	Send(ctx context.Context, ev *event.SendEvent) error
	// Close 关闭连接
	Close()
	// Connect 建立连接
	Connect(ctx context.Context, token string) error
	// On 注册事件监听器
	On(event string, listener event.Listener)
}

type client struct {
	eioParser  eioparser.Parser
	sioEncoder sioparser.Encoder
	sioDecoder sioparser.Decoder

	conn    *websocket.Conn
	readCh  chan *sioparser.Packet
	writeCh chan *packet.Packet

	opts  Options
	token string

	listeners sync.Map

	cmu         sync.Mutex
	connCh      chan struct{}
	closeCh     chan struct{}
	isConnected atomic.Bool
}

// NewClient 构造会话客户端
func NewClient(opts ...Option) Client {
	options := DefaultOptions()
	for _, f := range opts {
		f(&options)
	}
	fmt.Printf("Options:%+v\n", options)
	c := &client{
		opts:       options,
		eioParser:  eioparser.Parserv4(),
		sioEncoder: sioparser.NewEncoder(),
		sioDecoder: sioparser.NewDecoder(),
		listeners:  sync.Map{},
		closeCh:    make(chan struct{}),
	}
	c.sioDecoder.On("decoded", c.onDecoded)
	return c
}

// On 注册事件监听器
func (c *client) On(event string, listener event.Listener) {
	c.listeners.Store(event, listener)
}

func (c *client) onDecoded(args ...any) {
	for _, arg := range args {
		if p, ok := arg.(*sioparser.Packet); ok {
			if p.Type == sioparser.Connect {
				c.connCh <- struct{}{}
			} else {
				c.readCh <- p
			}
		} else {
			log.Printf("onDecoded receives a malformed packet: %s", spew.Sdump(arg))
		}
	}
}

func (c *client) write() {
	defer close(c.writeCh)
	for {
		select {
		case <-c.closeCh:
			return
		case p := <-c.writeCh:
			buf, _ := c.eioParser.EncodePacket(p, true)
			c.conn.WriteMessage(websocket.TextMessage, buf.Bytes())
		}
	}
}

func (c *client) read() {
	defer close(c.readCh)
	for {
		select {
		case <-c.closeCh:
			return
		default:
			typ, message, err := c.conn.ReadMessage()
			if err != nil {
				log.Printf("read: read message error: %+v", err)
				c.Close()
				return
			}

			p, err := c.eioParser.DecodePacket(types.NewStringBuffer(message))
			if err != nil {
				log.Printf("read: decode eio packet error: %+v", err)
				c.Close()
				return
			}

			if c.opts.Debug {
				log.Printf("read: wsType: %+v, eioPacket: %+v", typ, p)
			}
			switch p.Type {
			case packet.Ping:
				c.writeCh <- &packet.Packet{Type: packet.Pong}
			case packet.Message:
				c.sioDecoder.Add(p.Data)
			case packet.Open:
				if c.token != "" {
					buf := c.sioEncoder.Encode(&sioparser.Packet{Type: sioparser.Connect, Data: auth{Token: c.token}})
					c.writeCh <- &packet.Packet{Type: packet.Message, Data: buf[0]}
				}
			default:
			}
		}
	}
}

// GetToken 获取连接令牌
func (c *client) GetToken(ctx context.Context, req *GetTokenReq) (string, error) {
	crendential := common.NewCredential(req.SecretID, req.SecretKey)
	clientProfile := profile.NewClientProfile()
	clientProfile.HttpProfile.RootDomain = req.TencentCloudDomain
	clientProfile.HttpProfile.Scheme = req.Scheme
	clientProfile.HttpProfile.ReqMethod = req.ReqMethod
	client, err := v20231130.NewClient(crendential, req.Region, clientProfile)
	if err != nil {
		log.Fatalf("get tencent client error: %+v", err)
	}
	connType := int64(req.Type)
	botAppKey := req.BotAppKey
	visitorBizID := req.VisitorBizID
	labels := make([]*v20231130.GetWsTokenReq_Label, 0)
	for _, label := range req.VisitorLabels {
		values := make([]*string, 0)
		for _, value := range label.Values {
			values = append(values, &value)
		}
		labels = append(labels, &v20231130.GetWsTokenReq_Label{
			Name:   &label.Name,
			Values: values,
		})
	}
	tokenReq := v20231130.NewGetWsTokenRequest()
	tokenReq.Type = &connType
	tokenReq.BotAppKey = &botAppKey
	tokenReq.VisitorBizId = &visitorBizID
	tokenReq.VisitorLabels = labels

	tokenCtx, _ := context.WithTimeout(ctx, 10*time.Second)
	tokenRsp, err := client.GetWsTokenWithContext(tokenCtx, tokenReq)
	if err != nil {
		log.Fatalf("get token error: %+v", err)
	}
	log.Printf("tokenRsp:%+v",tokenRsp)
	return *tokenRsp.Response.Token, nil
}

// Send 发送消息
func (c *client) Send(ctx context.Context, ev *event.SendEvent) error {
	if !c.isConnected.Load() {
		return fmt.Errorf("send: client is disconnected")
	}
	payload, _ := json.Marshal(ev)
	buf := c.sioEncoder.Encode(&sioparser.Packet{
		Type: sioparser.Event,
		Data: []any{ev.Name(), event.EventWrapper{Payload: payload}},
	})
	c.writeCh <- &packet.Packet{Type: packet.Message, Data: buf[0]}
	return nil
}

// Connect 建立连接
func (c *client) Connect(ctx context.Context, token string) (err error) {
	if c.isConnected.Load() {
		return nil
	}

	c.cmu.Lock()
	defer c.cmu.Unlock()

	c.token = token
	c.readCh = make(chan *sioparser.Packet, c.opts.ReadChannelSize)
	c.connCh = make(chan struct{}, 1)
	c.writeCh = make(chan *packet.Packet, c.opts.WriteChannelSize)
	c.closeCh = make(chan struct{})
	if c.conn, _, err = websocket.DefaultDialer.DialContext(ctx, c.opts.ConnEndpoint, nil); err != nil {
		return err
	}

	go c.read()
	go c.write()
	go c.dispatch()

	for {
		select {
		case <-ctx.Done():
			return ctx.Err()
		case <-c.closeCh:
			return nil
		case <-c.connCh:
			c.isConnected.Store(true)
			return nil
		}
	}
}

// Close 关闭连接
func (c *client) Close() {
	c.conn.Close()
	close(c.closeCh)
	c.isConnected.Store(false)
}

func (c *client) dispatch() error {
	for {
		select {
		case <-c.closeCh:
			return nil
		case p, ok := <-c.readCh:
			if !ok {
				return nil
			}
			switch p.Type {
			case sioparser.Event:
				args, ok := p.Data.([]any)
				if !ok || len(args) < 2 {
					log.Printf("dispatch: receive malformed event: %+v", p.Data)
					continue
				}
				name, ok := args[0].(string)
				if !ok {
					log.Printf("dispatch: receive malformed event: %+v", p.Data)
					continue
				}
				data, _ := json.Marshal(args[1])
				ev := event.EventWrapper{}
				if err := json.Unmarshal(data, &ev); err != nil {
					log.Printf("dispatch: unmarshal event body error: %+v, data: %+v", err, data)
					continue
				}
				if h, ok := c.listeners.Load(name); ok {
					h.(event.Listener)(ev)
				}
			default:
			}
		}
	}
}
