// Package sse is a chat client based on sse.
package sse

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"sync"
	"time"

	"bufio"
	"io"
	"strings"

	"chat-demo-client-golang/event"
)

// Client 会话客户端
type Client interface {
	// On 注册事件监听器
	On(event string, listener event.Listener)
	// Send 发送消息
	Send(ctx context.Context, ev *event.SseSendEvent) error
}

type client struct {
	opts Options

	listeners sync.Map
}

// NewClient 构造会话客户端
func NewClient(opts ...Option) Client {
	options := DefaultOptions()
	for _, f := range opts {
		f(&options)
	}
	fmt.Printf("Options:%+v\n", options)
	return &client{
		opts:      options,
		listeners: sync.Map{},
	}
}

// On 注册事件监听器
func (c *client) On(event string, listener event.Listener) {
	c.listeners.Store(event, listener)
}

// Send 发送消息
func (c *client) Send(ctx context.Context, ev *event.SseSendEvent) error {
	//  创建和配置 HTTP 请求
	body, err := json.Marshal(ev)
	if err != nil {
		return fmt.Errorf("failed to marshal request body: %w", err)
	}

	req, err := http.NewRequestWithContext(ctx, http.MethodPost, c.opts.ConnEndpoint, bytes.NewReader(body))
	if err != nil {
		return fmt.Errorf("failed to create request: %w", err)
	}

	// 设置 SSE 请求头
	req.Header.Set("Content-Type", "application/json")
	req.Header.Set("Accept", "text/event-stream")
	req.Header.Set("Cache-Control", "no-cache")
	req.Header.Set("Connection", "keep-alive")

	//在发送请求前记录开始时间
	startTime := time.Now() // 发送请求和基础响应处理

	httpClient := &http.Client{}
	resp, err := httpClient.Do(req)
	if err != nil {
		return fmt.Errorf("request failed: %w", err)
	}

	defer resp.Body.Close()

	if resp.StatusCode != http.StatusOK {
		// 读取一些响应体内容以获取更多错误信息
		bodyBytes, _ := io.ReadAll(io.LimitReader(resp.Body, 4096))
		return fmt.Errorf("bad response status: %s, body: %s", resp.Status, string(bodyBytes))
	}

	//使用 bufio.Scanner 流式读取和解析响应
	scanner := bufio.NewScanner(resp.Body)

	// 创建一个足够大的缓冲区。
	const maxCapacity = 1 * 1024 * 1024 // 1 MB
	buf := make([]byte, maxCapacity)
	scanner.Buffer(buf, maxCapacity)

	var eventName string
	var dataBuffer bytes.Buffer

	for scanner.Scan() {

		line := scanner.Text()
		fmt.Printf("%s\n", line)
		// SSE 消息格式解析
		if line == "" {
			if dataBuffer.Len() > 0 {

				if eventName == "" {
					eventName = "message" //默认为 "message"
				}

				if h, ok := c.listeners.Load(eventName); ok {
					var eventData event.EventWrapper
					// 去掉 dataBuffer 末尾可能多余的换行符
					dataBytes := bytes.TrimSuffix(dataBuffer.Bytes(), []byte("\n"))
					if err := json.Unmarshal(dataBytes, &eventData); err != nil {

						fmt.Printf("Error unmarshalling data for event '%s': %v\n", eventName, err)

					} else {
						h.(event.Listener)(eventData)
					}
				}

				// 重置，为下一个事件做准备
				dataBuffer.Reset()
				eventName = ""
			}
			continue
		}

		// 解析 "field: value" 格式的行
		if key, value, found := strings.Cut(line, ":"); found {
			// 去空格
			value = strings.TrimPrefix(value, " ")

			switch key {
			case "event":
				eventName = value
			case "data":
				// 多行 data 字段需要拼接，并在它们之间加上换行符
				dataBuffer.WriteString(value)
				dataBuffer.WriteString("\n")

				var data map[string]interface{}
				if err := json.Unmarshal([]byte(value), &data); err == nil {
					// 检查 payload 是否存在
					if payload, ok := data["payload"].(map[string]interface{}); ok {
						// 检查 is_final 和 is_from_self 条件
						if isFinal, ok := payload["is_final"].(bool); ok && isFinal {
							if isFromSelf, ok := payload["is_from_self"].(bool); ok && !isFromSelf {
								// 计算首包耗时
								hs := time.Since(startTime).Seconds()
								fmt.Printf("总耗时：%v秒\n", hs)
							}
						}

						if isFromSelf, ok := payload["is_from_self"].(bool); ok && isFromSelf {

							hs := time.Since(startTime).Seconds()
							fmt.Printf("第一个网络包耗时：%v秒\n", hs)
						}

					}
				}

			}
		}
	}

	if err := scanner.Err(); err != nil {
		if ctx.Err() != nil {
			fmt.Printf("scanner error: %v\n", ctx.Err())
			return nil
		}
		fmt.Println("读取流失败:", err)
	}

	return nil
}
