package main

import (
	"chat-demo-client-golang/event"
	"chat-demo-client-golang/sse"
	"chat-demo-client-golang/websocket"
	"context"
	"encoding/json"
	"log"
	"time"
	"crypto/rand"
	"fmt"
	"github.com/google/uuid"
)

const (
	question          = "你是谁"
	streamingThrottle = 5 // 节流控制
)

const (
	botAppKey    = "" // 一般为8位字符串，具体可参考 https://cloud.tencent.com/document/product/1759/109469#4eecb8c1-6ce4-45f5-8fa2-b269449d8efa
	visitorBizID = "" // 该字段客户可自行选择启用，规则： string(64)

	tencentCloudDomain = "tencentcloudapi.com" // 腾讯云API域名
	scheme             = "https"               // 协议
	reqMethod          = "POST"                // 请求方法
	region             = "ap-guangzhou"        // 地域

	secretID  = "" // 填入腾讯云AKSK密钥(从腾讯云控制台获取)
	secretKey = "" // 填入腾讯云AKSK密钥(从腾讯云控制台获取)

	sseUrl       = "https://wss.lke.cloud.tencent.com/v1/qbot/chat/sse"
	webSocketUrl = "wss://wss.lke.cloud.tencent.com/v1/qbot/chat/conn/?EIO=4&transport=websocket"
)

// labels 访客标签
// var labels = []event.Label{{Name: "car", Values: []string{"c1"}}} // 标签根据需要设置，可以不设置
var labels []event.Label

var sessionID string // 会话 ID，用于标识一个会话（外部系统提供，建议不同的用户端会话传入不同的 session_id，否则同一个应用下的不同用户的消息记录会串掉）
var requestID string // 请求 ID，用于标识一个请求（作消息串联，建议每个请求使用不同的 request_id）

func main() {
	sseExample()
	//websocketExample()
	time.Sleep(5000 * time.Second)
}

func onReply(ev event.EventWrapper) {
	reply := event.ReplyEvent{}
	if err := json.Unmarshal(ev.Payload, &reply); err != nil {
		log.Printf("unmarshal reply event error: %v", err)
		return
	}
	//log.Printf("Reply: %+v", reply)
}

func sseExample() {
	ctx := context.TODO()
	option := sse.WithConnEndpoint(sseUrl)
	c := sse.NewClient(option)
	c.On(event.EventReply, onReply)
	sessionID = getSessionID()
	c.Send(ctx, &event.SseSendEvent{
		Content:           question,
		BotAppKey:         botAppKey,
		VisitorBizID:      visitorBizID,
		SessionID:         sessionID,
		VisitorLabels:     labels,
		StreamingThrottle: streamingThrottle,
	})
}

func websocketExample() {
	ctx := context.TODO()
	option := websocket.WithConnEndpoint(webSocketUrl)
	c := websocket.NewClient(option)
	c.On(event.EventReply, onReply)
	token, err := c.GetToken(ctx, &websocket.GetTokenReq{
		Type:               websocket.ConnTypeAPI,
		BotAppKey:          botAppKey,
		VisitorBizID:       visitorBizID,
		VisitorLabels:      labels,
		TencentCloudDomain: tencentCloudDomain,
		Scheme:             scheme,
		ReqMethod:          reqMethod,
		SecretID:           secretID,
		SecretKey:          secretKey,
		Region:             region,
	})
	if err != nil {
		log.Fatalf("get token error: %+v", err)
	}

	if err := c.Connect(ctx, token); err != nil {
		log.Fatalf("connect error: %+v", err)
	}
	sessionID = getSessionID()
	requestID = getRequestID()
	c.Send(ctx, &event.SendEvent{
		SessionID:         sessionID,
		Content:           question,
		RequestID:         requestID,
		StreamingThrottle: streamingThrottle,
	})
}


const sessionIDLength = 24

func getRequestID() string {
	b := make([]byte, sessionIDLength)
	_, err := rand.Read(b)
	if err != nil {
		return ""
	}

	return fmt.Sprintf("%x", b)
}

func getSessionID() string {
	return uuid.NewString()
}
